/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CFileImpWin32.cpp,v 1.4 2000/05/13 06:34:21 nryan Exp $
____________________________________________________________________________*/

#include "pgpClassesConfig.h"
#include <winioctl.h>

#include "UMath.h"
#include "CFileImpWin32.h"

_USING_PGP

// Class CFileImpWin32 member functions

CFileImpWin32::~CFileImpWin32()
{
	try
	{
		if (IsOpened())
			Close();
	}
	catch (CComboError&) { }
}

PGPUInt64 
CFileImpWin32::GetLength() const
{
	pgpAssert(IsOpened());

	ULARGE_INTEGER	liLength;
	liLength.LowPart = GetFileSize(mFileHandle, &liLength.HighPart);

	if ((liLength.LowPart == 0xFFFFFFFF) && (GetLastError() != NO_ERROR))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());

	return liLength.QuadPart;
}

PGPUInt64 
CFileImpWin32::GetUniqueFileId() const
{
	pgpAssert(IsOpened());

	BY_HANDLE_FILE_INFORMATION	fileInfo;

	if (!GetFileInformationByHandle(mFileHandle, &fileInfo))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());

	return UMath::MakeQWord(fileInfo.nFileIndexLow, fileInfo.nFileIndexHigh);
}

void 
CFileImpWin32::SetIsCompressed(PGPBoolean isCompressed)
{
	pgpAssert(IsOpened());

	PGPUInt16	compressState	= (isCompressed ? 
		COMPRESSION_FORMAT_DEFAULT : COMPRESSION_FORMAT_NONE);
	ULONG		bytesReturned;

	if (!DeviceIoControl(mFileHandle, FSCTL_SET_COMPRESSION, &compressState, 
		sizeof(compressState), NULL, 0, &bytesReturned, NULL))
	{
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
	}
}

void 
CFileImpWin32::SetLength(PGPUInt64 length)
{
	pgpAssert(IsOpened());

	EzSetFilePointer(length);

	if (SetEndOfFile(mFileHandle) == 0)
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
}

void 
CFileImpWin32::Flush()
{
	pgpAssert(IsOpened());

	if (!FlushFileBuffers(mFileHandle))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
}

void 
CFileImpWin32::Open(const char *path, PGPUInt32 flags)
{
	pgpAssert(!IsOpened());
	pgpAssertStrValid(path);

	mPath.Assign(path);

	// Set access flags.
	PGPUInt32	accessMask	= GENERIC_READ;

	if (!(flags & CFile::kReadOnlyFlag))
		accessMask |= GENERIC_WRITE;

	// Set share flags.
	PGPUInt32	shareAccess	= NULL;

	if (!(flags & CFile::kDenyReadFlag))
		shareAccess |= FILE_SHARE_READ;

	if (flags & CFile::kShareWriteFlag)
		shareAccess |= FILE_SHARE_WRITE;

	// Set the create disposition.
	PGPUInt32	createDisp;

	if (flags & CFile::kCreateIfFlag)
		createDisp = OPEN_ALWAYS;
	else
		createDisp = OPEN_EXISTING;

	// Set the file flags.
	PGPUInt32	fileFlags	= FILE_FLAG_SEQUENTIAL_SCAN;

	if (flags & CFile::kNoBufferingFlag)
		fileFlags |= FILE_FLAG_NO_BUFFERING;

	// Open the file handle.
	mFileHandle = CreateFile(mPath, accessMask, shareAccess, NULL, 
		createDisp, fileFlags, NULL);

	if (mFileHandle == INVALID_HANDLE_VALUE)
		THROW_ERRORS(kPGPError_CantOpenFile, GetLastError());

	mIsOpened = TRUE;
	mOpenFlags = flags;
}

void 
CFileImpWin32::Close()
{
	pgpAssert(IsOpened());

	if (!CloseHandle(mFileHandle))
	{
		pgpAssert(FALSE);
	}

	mIsOpened	= FALSE;
	mFileHandle	= INVALID_HANDLE_VALUE;
	mOpenFlags	= CFile::kNoFlags;
}

void 
CFileImpWin32::Delete(const char *path)
{
	pgpAssertStrValid(path);

	if (!DeleteFile(path))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
}

void 
CFileImpWin32::Move(const char *oldPath, const char *newPath)
{
	pgpAssertStrValid(oldPath);
	pgpAssertStrValid(newPath);

	if (!MoveFile(oldPath, newPath))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
}

void 
CFileImpWin32::Read(void *buf, PGPUInt64 pos, PGPUInt32 nBytes) const
{
	pgpAssert(IsOpened());
	pgpAssertAddrValid(buf, VoidAlign);

	EzSetFilePointer(pos);

	unsigned long numRead;

	if (!ReadFile(mFileHandle, buf, nBytes, &numRead, NULL))
		THROW_ERRORS(kPGPError_ReadFailed, GetLastError());
}

void 
CFileImpWin32::Write(const void *buf, PGPUInt64 pos, PGPUInt32 nBytes) const
{
	pgpAssert(IsOpened());
	pgpAssertAddrValid(buf, VoidAlign);

	EzSetFilePointer(pos);

	unsigned long numWritten;

	if (!WriteFile(mFileHandle, buf, nBytes, &numWritten, NULL))
		THROW_ERRORS(kPGPError_WriteFailed, GetLastError());
}

void 
CFileImpWin32::EzSetFilePointer(PGPUInt64 bytes) const
{
	pgpAssert(IsOpened());

	LARGE_INTEGER	liBytes;
	liBytes.QuadPart = static_cast<PGPInt64>(bytes);

	liBytes.LowPart = SetFilePointer(mFileHandle, liBytes.LowPart, 
		&liBytes.HighPart, FILE_BEGIN);
	
	if ((liBytes.LowPart == 0xFFFFFFFF) && (GetLastError() != NO_ERROR))
		THROW_ERRORS(kPGPError_FileOpFailed, GetLastError());
}
